'use strict';

/* --------------------------------------------------------------
 datatable_checkbox_mapping.js 2016-10-18
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## DataTable Checkbox Mapping Extension
 *
 * This extension maps the bulk actions from a datatable dropdown to the selected checkbox carets. Bind this
 * extension into a datatable element which has a first.
 *
 * ### Options
 *
 * **Bulk Action Selector | `data-datatable_checkbox_mapping-bulk-action` | String | Optional**
 *
 * Provide a selector for the bulk action dropdown widget. Default value is '.bulk-action'.
 *
 * **Bulk Selection Checkbox Selector | `data-datatable_checkbox_mapping-bulk-selection` | String | Optional**
 *
 * Provide a selector for the bulk selection checkbox in the table header. Default value is '.bulk-selection'.
 *
 * **Row Selection Checkbox Selector | `data-datatable_checkbox_mapping-row-selection` | String | Optional**
 *
 * Provide a selector for the row selection checkbox in the table body. Default value is 'tbody tr input:checkbox'.
 *
 * **Caret Icon Class | `data-datatable_checkbox_mapping-caret-icon-class` | String | Optional**
 *
 * Provide a FontAwesome icon class for the checkbox caret. Default value is 'fa-caret-down'. Provide only the class
 * name without dots or the "fa" class.
 *
 * @module Admin/Extensions/datatable_checkbox_mapping
 */
gx.extensions.module('datatable_checkbox_mapping', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Default Options
     *
     * @type {Object}
     */
    var defaults = {
        bulkAction: '.bulk-action',
        bulkSelection: '.bulk-selection',
        caretIconClass: 'fa-caret-down',
        rowSelection: 'tbody tr input:checkbox'
    };

    /**
     * Final Options
     *
     * @type {Object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Bulk Action Selector
     *
     * @type {jQuery}
     */
    var $bulkAction = $(options.bulkAction);

    /**
     * Bulk Selection Selector
     *
     * @type {jQuery}
     */
    var $bulkSelection = $this.find(options.bulkSelection).last();

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Toggle the dropdown menu under the caret.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _toggleDropdownMenu(event) {
        event.stopPropagation();
        event.preventDefault();

        if ($bulkAction.hasClass('open')) {
            $bulkAction.removeClass('open');
            return;
        }

        var caretPosition = $(event.target).offset();
        var $dropdownMenu = $bulkAction.find('.dropdown-menu');

        // Open the dropdown menu.
        $bulkAction.addClass('open');

        // Reposition the dropdown menu near the clicked caret.
        $dropdownMenu.offset({
            top: caretPosition.top + 16,
            left: caretPosition.left
        });

        // Don't show the long empty dropdown menu box when it is repositioned.
        $dropdownMenu.css({ bottom: 'initial' });

        // Show the dropdown menu under or above the caret, depending on the viewport.
        if (_dropdownIsOutOfView($dropdownMenu)) {
            $dropdownMenu.offset({
                top: caretPosition.top - $dropdownMenu.outerHeight(),
                left: caretPosition.left
            });
        }
    }

    /**
     * Reset the dropdown position to its original state.
     */
    function _resetDropdownPosition() {
        $bulkAction.find('.dropdown-menu').css({
            top: '',
            left: '',
            bottom: ''
        });
    }

    /**
     * Add a caret to the table head checked checkbox.
     */
    function _addCaretToBulkSelection() {
        var $th = $bulkSelection.parents('th');

        if ($th.find('.' + options.caretIconClass).length === 0) {
            $th.append('<i class="fa ' + options.caretIconClass + '"></i>');
        }
    }

    /**
     * Remove the caret from the bulk selection checkbox.
     */
    function _removeCaretFromBulkSelection() {
        $bulkSelection.parents('th').find('.' + options.caretIconClass).remove();
    }

    /**
     * Add a caret to the checked checkbox.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _addCaretToActivatedCheckbox(event) {
        $(event.target).parents('td').append('<i class="fa ' + options.caretIconClass + '"></i>');
    }

    /**
     * Remove the caret from the checkbox if the checkbox is unchecked.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _removeCaretFromCheckbox(event) {
        $(event.target).parents('tr').find('.' + options.caretIconClass).remove();
    }

    /**
     * Start listening for click events for the caret symbol.
     *
     * When the caret symbol gets clicked, show the dropdown menu.
     */
    function _listenForCaretClickEvents() {
        $this.find('tr .' + options.caretIconClass).off('click').on('click', _toggleDropdownMenu);
    }

    /**
     * Set the bulk selection state.
     *
     * @param {Boolean} isChecked Whether the checkbox will be checked or not.
     */
    function _setBulkSelectionState(isChecked) {
        $bulkSelection.prop('checked', isChecked);

        if (isChecked) {
            $bulkSelection.parents('.single-checkbox').addClass('checked');
            _addCaretToBulkSelection();
            _listenForCaretClickEvents();
        } else {
            $bulkSelection.parents('.single-checkbox').removeClass('checked');
            _removeCaretFromBulkSelection();
        }
    }

    /**
     * Checks if the provided dropdown is outside of the viewport (in height).
     *
     * @param {jQuery} $dropdownMenu Dropdown menu selector.
     *
     * @return {Boolean}
     */
    function _dropdownIsOutOfView($dropdownMenu) {
        var dropDownMenuOffset = $dropdownMenu.offset().top + $dropdownMenu.outerHeight() + 50;
        var windowHeight = window.innerHeight + $(window).scrollTop();
        return dropDownMenuOffset > windowHeight;
    }

    /**
     * On Single Checkbox Ready Event
     *
     * Bind the checkbox mapping functionality on the table. We need to wait for the "single_checkbox:ready",
     * that will be triggered with every table re-draw. Whenever a row checkbox is clicked the bulk-action
     * caret icon will be added next to it.
     */
    function _onSingleCheckboxReady() {
        // Find all checkboxes table body checkboxes.
        var $tableBodyCheckboxes = $this.find(options.rowSelection);

        // Table data checkbox event handling.
        $tableBodyCheckboxes.on('change', function (event) {
            // Close any open dropdown menus.
            $bulkAction.removeClass('open');

            if ($(event.target).prop('checked')) {
                _addCaretToActivatedCheckbox(event);
                _listenForCaretClickEvents();
            } else {
                _removeCaretFromCheckbox(event);
            }

            // Activate the table head checkbox if all checkboxes are activated. Otherwise deactivate it.
            _setBulkSelectionState($tableBodyCheckboxes.not(':checked').length === 0);
        });
    }

    /**
     * Add or remove the caret from the table head checkbox.
     *
     * @param {jQuery.Event} event
     */
    function _onBulkSelectionChange(event) {
        if ($bulkSelection.parents('.single-checkbox').length === 0) {
            return; // Do not proceed with the function if the thead single-checkbox is not ready yet.
        }

        if ($bulkSelection.prop('checked')) {
            _addCaretToBulkSelection();
            _listenForCaretClickEvents();
        } else {
            _removeCaretFromBulkSelection(event);
        }
    }

    /**
     *  Event handling for the original dropdown button click.
     */
    function _onBulkActionDropdownToggleClick() {
        _resetDropdownPosition();
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('single_checkbox:ready', _onSingleCheckboxReady);
        $bulkSelection.on('change', _onBulkSelectionChange);
        $bulkAction.find('.dropdown-toggle').on('click', _onBulkActionDropdownToggleClick);
        done();
    };
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
